/* Copyright (c) 1998 Roger Lindsj */

package se.chalmers.dtek.d97roli.connect4;
// Gui for Connect4 game.
import java.awt.*;
import java.applet.*;
import java.awt.event.*;
import java.io.*;

public class Connect4 extends Applet implements MouseListener, MouseMotionListener, ActionListener{

  // board
  private Arena arena;
  // computer avaluater
  protected ComputerPlayer computer;
  
  private Image bgImage = null;
  private Graphics bgGraphics = null;

  private int tileWidth = 0;
  private int tileHeight = 0;
  private int pointerCol = 3;
  private Image pointer = null;
  private Image emptyTile = null;
  private Image player1Tile = null;
  private Image player2Tile = null;
  private Image winRed = null;
  private Image winYellow = null;
  private Image winDraw = null;

  private MenuItem undo;
  
  private MenuItem computermove;
  private MenuItem humanVsHuman;
  private MenuItem humanVsComputer;
  private MenuItem computerVsHuman;
  private MenuItem computerVsComputer;

  private MenuItem noviceLevel;
  private MenuItem intermediateLevel;
  
  private MenuItem reset;

  private PopupMenu pum;

  private boolean lastWasPopup = false;
  
  // default setting, human vs computer
  private int player1 = 0;   // default human vs computer
  private int player2 = 1;
  
  public void init(){

    // create an arena
    arena = new Arena(6,7);
    // get images
    try{
      InputStream imageStream;
      Toolkit tk = Toolkit.getDefaultToolkit();
      MediaTracker mt = new MediaTracker(this);
      
      byte[] imageBytes;
      imageStream = getClass().getResourceAsStream("black_tile.gif");
      imageBytes = new byte[imageStream.available()];
      imageStream.read(imageBytes);
      emptyTile = tk.createImage(imageBytes); 
      mt.addImage(emptyTile,0);
      
      imageStream = getClass().getResourceAsStream("red_tile.gif");
      imageBytes = new byte[imageStream.available()];
      imageStream.read(imageBytes);
      player1Tile = tk.createImage(imageBytes); 
      mt.addImage(player1Tile,0);
      
      imageStream = getClass().getResourceAsStream("yellow_tile.gif");
      imageBytes = new byte[imageStream.available()];
      imageStream.read(imageBytes);
      player2Tile = tk.createImage(imageBytes); 
      mt.addImage(player2Tile,0);
      
      imageStream = getClass().getResourceAsStream("down.gif");
      imageBytes = new byte[imageStream.available()];
      imageStream.read(imageBytes);
      pointer = tk.createImage(imageBytes); 
      mt.addImage(pointer,0);
      
      imageStream = getClass().getResourceAsStream("red_win.gif");
      imageBytes = new byte[imageStream.available()];
      imageStream.read(imageBytes);
      winRed = tk.createImage(imageBytes); 
      mt.addImage(winRed,0);
      
      imageStream = getClass().getResourceAsStream("yellow_win.gif");
      imageBytes = new byte[imageStream.available()];
      imageStream.read(imageBytes);
      winYellow = tk.createImage(imageBytes);
      mt.addImage(winYellow,0);
      
      imageStream = getClass().getResourceAsStream("draw.gif");
      imageBytes = new byte[imageStream.available()];
      imageStream.read(imageBytes);
      winDraw = tk.createImage(imageBytes); 
      mt.addImage(winDraw,0);

      mt.waitForAll();
      
      tileWidth=emptyTile.getWidth(null);
      tileHeight=emptyTile.getHeight(null);
      bgImage = createImage(tileWidth*arena.width,tileHeight*arena.height);
      bgGraphics = bgImage.getGraphics();

    }catch(Exception e){e.printStackTrace();};
    
    pum = new PopupMenu("Connect 4");

    undo = new MenuItem("Undo");
    pum.add(undo);
    computermove = new MenuItem("Make best move");
    pum.add(computermove);
    
    pum.add(new MenuItem("-"));
	    
    Menu gameType = new Menu("Gametype");    
    humanVsHuman = new MenuItem("Human vs Human");
    gameType.add(humanVsHuman);
    humanVsComputer = new MenuItem("Human vs Computer");
    gameType.add(humanVsComputer);
    computerVsHuman = new MenuItem("Computer vs Human");
    gameType.add(computerVsHuman);
    computerVsComputer = new MenuItem("Computer vs Computer");
    gameType.add(computerVsComputer);
    pum.add(gameType);

    Menu level = new Menu("Game level");
    noviceLevel = new MenuItem("Novice");
    level.add(noviceLevel);
    intermediateLevel = new MenuItem("Intermediate");
    level.add(intermediateLevel);
    pum.add(level);

    pum.add(new MenuItem("-"));

    reset = new MenuItem("Reset");

    pum.add(reset);

    undo.addActionListener(this);
    computermove.addActionListener(this);
    humanVsHuman.addActionListener(this);
    humanVsComputer.addActionListener(this);
    computerVsHuman.addActionListener(this);
    computerVsComputer.addActionListener(this);
    noviceLevel.addActionListener(this);
    intermediateLevel.addActionListener(this);
    reset.addActionListener(this);

    add(pum);
    addMouseListener(this);
    addMouseMotionListener(this);

    // create computer evaluater
    computer = new ComputerPlayer(arena);
    computer.addActionListener(this);
  }
  
  public void start(){
    // paint us
    updateAllTiles();
    updatePointer(pointerCol);
    repaint();
  }
  
  public void stop(){
  }

  public void updateAllTiles(){
    for(int i=0; i<arena.width; i++)
      updateCol(i);
  }
  
  public void updateCol(int col){
    for(int row=0; row<arena.height;row++)
      updateTile(row,col);
  }
  
  public void updateTile(int row, int col){
    if(bgGraphics!=null)
      switch(arena.tokens[row][col]){
      case -1:
	bgGraphics.drawImage(player1Tile,tileWidth*col,tileHeight*row,null);
	break;
      case 0:
	bgGraphics.drawImage(emptyTile,tileWidth*col,tileHeight*row,null);
	break;
      case 1:
	bgGraphics.drawImage(player2Tile,tileWidth*col,tileHeight*row,null);
	break;
      }
  }

  public void updatePointer(int col){
    if(bgGraphics!=null)
      bgGraphics.drawImage(pointer,tileWidth*col+(tileWidth-pointer.getWidth(null))/2,0,null);
  }

  public void updateWinner(){
    if(arena.player1_patterns[arena.player1_patterns.length-1]>0)
      bgGraphics.drawImage(winRed,(bgImage.getWidth(null)-winRed.getWidth(null))/2,(bgImage.getHeight(null)-winRed.getHeight(null))/2,null);
    else if(arena.player2_patterns[arena.player2_patterns.length-1]>0)
      bgGraphics.drawImage(winYellow,(bgImage.getWidth(null)-winYellow.getWidth(null))/2,(bgImage.getHeight(null)-winYellow.getHeight(null))/2,null);
    else
      bgGraphics.drawImage(winDraw,(bgImage.getWidth(null)-winDraw.getWidth(null))/2,(bgImage.getHeight(null)-winDraw.getHeight(null))/2,null);
  }
      

  public void paint(Graphics g){
    g.drawImage(bgImage,0,0,null);
  }

  public void update(Graphics g){
    paint(g);
  }  

  // listen to menu and computerplayer
  public void actionPerformed(ActionEvent ae){
    Object o = ae.getSource();
    boolean makemove=false;;
    if(o==undo){
      arena.removeLastToken();
      updateAllTiles();
    }else if(o==reset){
      arena.reset();
      updateAllTiles();
      makemove=true;
    }else if(o==humanVsHuman){
      player1=0;
      player2=0;
    }else if(o==humanVsComputer){
      player1=0;
      player2=1;
      makemove=true;
    }else if(o==computerVsHuman){
      player1=1;
      player2=0;
      makemove=true;
    }else if(o==computerVsComputer){
      player1=1;
      player2=1;
      makemove=true;
    }else if(o==computermove){
      if(!arena.gotWinner())
	computerMove();
    }else if(o==computer){
      int col=arena.moves[arena.movesLeft];
      updateCol(col);
      if(col==pointerCol)
	updatePointer(col);
      makemove=true;
    }else if(o==noviceLevel){
      computer.setLevel(ComputerPlayer.NOVICE);
    }else if(o==intermediateLevel){
      computer.setLevel(ComputerPlayer.INTERMEDIATE);
    }
    if(arena.gotWinner() || arena.movesLeft==0)
      updateWinner();
    repaint();

    // should the computer make next move?
    if(makemove)
      if(((arena.currentPlayer==-1)?player1:player2)==1)
	if(!arena.gotWinner() && arena.movesLeft>0)
	  computerMove();
  }
  
  public void mouseClicked(MouseEvent e){}

  public void setPopupMenu(){
    if(arena.movesLeft==arena.width*arena.height)
      undo.setEnabled(false);
    else
      undo.setEnabled(true);
    if(arena.movesLeft==0 || arena.gotWinner())
      computermove.setEnabled(false);
    else
      computermove.setEnabled(true);
    if(computer.maxDepth==ComputerPlayer.INTERMEDIATE){
      intermediateLevel.setEnabled(false);
      noviceLevel.setEnabled(true);
    }else{
      intermediateLevel.setEnabled(true);
      noviceLevel.setEnabled(false);
    }
    humanVsHuman.setEnabled(true);
    humanVsComputer.setEnabled(true);
    computerVsHuman.setEnabled(true);
    computerVsComputer.setEnabled(true);
    if(player1==0 && player2==0)
      humanVsHuman.setEnabled(false);
    else if(player1==0 && player2==1)
      humanVsComputer.setEnabled(false);
    else if(player1==1 && player2==0)
      computerVsHuman.setEnabled(false);
    else
      computerVsComputer.setEnabled(false);
  }
  
  public void mousePressed(MouseEvent e){
    if(e.isPopupTrigger()){
      lastWasPopup=true;
      setPopupMenu();
      pum.show(e.getComponent(),e.getX(),e.getY());
    }
  }
  
  public void mouseReleased(MouseEvent e){
    if(e.isPopupTrigger()){
      lastWasPopup=true;
      setPopupMenu();
      pum.show(e.getComponent(),e.getX(),e.getY());
    }else{
      // can we make the move we want?
      if(!arena.gotWinner() && ((arena.currentPlayer==-1)?player1:player2)==0 && !lastWasPopup || computer.isWorking){
	int row=arena.addToken(pointerCol,arena.currentPlayer);
	if(row>=0){
	  updateTile(row,pointerCol);
	  if(row==0)
	    updatePointer(pointerCol);
	  if(arena.gotWinner() || arena.movesLeft==0)
	    updateWinner();
	  repaint();
	  if(((arena.currentPlayer==-1)?player1:player2)==1)
	    if(!arena.gotWinner() && arena.movesLeft>0)
	      computerMove();
	}
      }
    }
    lastWasPopup=false;
  }
  
  public void mouseEntered(MouseEvent e){}
  
  public void mouseExited(MouseEvent e){}
  
  public void mouseDragged(MouseEvent e){}

  // draw the pointer
  public void mouseMoved(MouseEvent e){
    int x = e.getX()/tileWidth;
    if(x!=pointerCol && x>=0 && x<arena.width){
      updateTile(0,pointerCol);
      pointerCol=x;
      updatePointer(pointerCol);
      repaint();
    }
  }

  // start a thread to run the computer evaluater.
  public void computerMove(){
    (new Thread(){
      public void run(){
	computer.makeBestMove();
      }}).start();
  }
}
